<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Models\Device;
use App\Models\Label;
use App\Models\MessageLog;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;
use Str;

class WADeviceController extends Controller
{
    public function index(){
        return view('customer.wadevice.index');
    }

    public function getAll(){
        $customer = auth('customer')->user();
        if($customer->type=='staff'){
            $customer=$customer->staff;
        }

        $devices = $customer->devices()->where('device_type','whatsapp')->withCount(['sent_messages']);
        return datatables()->of($devices)
            ->addColumn('total_sent_message',function($q){
                return $q->sent_messages_count;
            })
            ->addColumn('status',function ($q){
                if ($q->status=='active'){
                    return '<span class="text-sm badge bg-success">Online</span>';
                }elseif($q->status=='inactive'){
                    return '<span class="text-sm badge bg-danger">Offline</span>';
                }
            })
            ->addColumn('action',function(Device $q) use($customer){
                $btn='';
               $user=auth('customer')->user();
               if($user->type=='staff') {
                   if ($user->hasPermissionTo('device_edit')) {
                    $btn = '<div class="btn-group">
                            <button type="button" class="btn btn-success rounded" data-toggle="dropdown" aria-expanded="false">
                                <i class="fas fa-ellipsis-v"></i>
                            </button>
                            <div class="dropdown-menu" role="menu" style="">

                            <a class="dropdown-item"  href="' . route('customer.wadevice.edit', [$q]) . '" >Edit</a>
                            </div>
                        </div>';
                   }

                   if ($user->hasPermissionTo('device_delete')) {
                    $btn = '<div class="btn-group">
                                        <button type="button" class="btn btn-success rounded" data-toggle="dropdown" aria-expanded="false">
                                            <i class="fas fa-ellipsis-v"></i>
                                        </button>
                                        <div class="dropdown-menu" role="menu" style="">
                                        <button class="btn btn-sm btn-danger" data-message="Are you sure you want to remove this device?"
                                        data-action=' . route('customer.wadevice.destroy', [$q]) . '
                                        data-input={"_method":"delete"}
                                        data-toggle="modal" data-target="#modal-confirm">Delete</button>
                                        </div>
                                    </div>';
                   }
               }else{
                $btn = '<div class="btn-group">
                <button type="button" class="btn btn-success rounded" data-toggle="dropdown" aria-expanded="false">
                    <i class="fas fa-ellipsis-v"></i>
                </button>
                <div class="dropdown-menu" role="menu" style="">

                <a class="dropdown-item"  href="' . route('customer.wadevice.edit', [$q]) . '" >Edit</a>
                    <button class="dropdown-item" data-message="Are you sure you want to remove this device?"
                    data-action=' . route('customer.wadevice.destroy', [$q]) . '
                    data-input={"_method":"delete"}
                    data-toggle="modal" data-target="#modal-confirm">Delete</button>
            </div>
        </div>';
               }

                return $btn ;
            })
            ->rawColumns(['status','action'])
            ->toJson();
    }

    public function create(){
        $customer = auth('customer')->user();
        if($customer->type=='staff'){
            $customer=$customer->staff;
        }
        if(auth('customer')->user()->type=='staff' && !$customer->hasPermissionTo('device_add')){
            return abort('404');
        }

        return view('customer.wadevice.create');
    }

    public function store(Request $request){

        $validator = Validator::make($request->all(), [
            'name' => 'required|unique:devices'
        ]);

        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->messages()], 400);
        }

        $customer = auth('customer')->user();

        $currentPlan = $customer->currentPlan();
        if (isset($currentPlan->renew_date) && $currentPlan->renew_date < Carbon::now()){
            return response()->json(['message' => 'Your Plan has expired'], 400);
        }

        $devices = Device::where('customer_id',$customer->id)->where('device_type', 'whatsapp')->count();
        if (($devices) >= $currentPlan->wa_device_limit) {
            return response()->json(['message' => 'Your have extended your Device limit'], 400);
        }

        $device = new Device();
        $device->customer_id = $customer->id;
        $device->name = $request->name;
        $device->device_unique_id = Str::random();
        $device->status = "inactive";
        $device->device_type = $request->type;
        $device->save();
        return response()->json(['message' => 'Device successfully added','data'=>['device_unique_id'=>$device->device_unique_id,'name'=>$device->name]], 201);
    }


    public function edit(Device $wadevice){
        $data['device']=$wadevice;

        if(auth('customer')->user()->type=='staff' && !auth('customer')->user()->hasPermissionTo('device_edit')){
            return abort('404');
        }
        return view('customer.wadevice.edit', $data);
    }

    public function update(Device $wadevice,Request $request){

        $request->validate([
            'name'=>'required'
        ]);

        if(auth('customer')->user()->type=='staff' && !auth('customer')->user()->hasPermissionTo('device_edit')){
            return abort('404');
        }

        $wadevice->update($request->only('name'));

        return redirect()->route('customer.wadevice.index')->with('success', 'Device successfully updated');
    }

    public function status(Request $request){
        $id= $request->id;
        $status= $request->status;
        $request->validate([
            'status'=>'required|in:active,inactive'
        ]);

        $customer = auth('customer')->user();
        if($customer->type=='staff'){
            $customer=$customer->staff;
        }

        $device = $customer->devices()->where('device_unique_id', $id)->orWhere('id',$id)->firstOrFail();
        $device->status= $status;
        $device->save();

        if($request->ajax()){
            return response()->json(['message' => 'Device status successfully changes']);
        }
        return redirect()->route('customer.wadevice.index')->with('success', 'Device status successfully changes');
    }

    public function destroy(Device $wadevice){
        $customer = auth('customer')->user();
        if($customer->type=='staff'){
            $customer=$customer->staff;
        }

        if(auth('customer')->user()->type=='staff' && !auth('customer')->user()->hasPermissionTo('device_delete')){
            return abort('404');
        }

        if ($customer->id != $wadevice->customer_id){
            return abort(404);
        }
        $wadevice->delete();

        return redirect()->route('customer.wadevice.index')->with('success', 'Device successfully deleted');
    }
}
