<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\CustomerPlan;
use App\Models\Device;
use App\Models\Message;
use App\Models\MessageLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class DeviceController extends Controller
{
    public function store(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'model' => 'required',
            'android_version' => 'required',
            'app_version' => 'required',
            'device_unique_id' => 'required|string',
            'sim_info' => 'required|array', // x:y:z => x=SimSlotIndex, y=SubscriberId, z=SimDisplayName
        ]);

        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->messages()], 400);
        }
        $user = auth()->user();
        if($user->type=='staff'){
            $user=$user->staff;
        }

        $existingDevice = Device::where('device_unique_id', $request->device_unique_id)->where('customer_id', $user->id)->first();
        if ($existingDevice) {
            $existingDevice->sim_info = json_encode($request->sim_info);
            $existingDevice->save();
            return response()->json(['message' => 'Device already added'], 200);
        }
        $customer = auth()->user();
        $currentPlan = $customer->currentPlan();
        if (isset($currentPlan->renew_date) && $currentPlan->renew_date < Carbon::now()){
            return response()->json(['message' => 'Your Plan has expired'], 400);
        }

        $devices = Device::where('customer_id',$user->id)->where('device_type', 'device')->count();
        $plan = $user->currentPlan();
        if (($devices) >= $plan->device_limit) {
            return response()->json(['message' => 'Your have extended your Device limit'], 400);
        }

        $device = new Device();
        $device->name = $request->name;
        $device->model = $request->model;
        $device->android_version = $request->android_version;
        $device->app_version = $request->app_version;
        $device->customer_id = $user->id;
        $device->status = 'active';
        $device->device_unique_id = $request->device_unique_id;
        $device->sim_info = json_encode($request->sim_info);
        $device->save();

        return response()->json(['message' => 'Device successfully added'], 201);
    }

    public function getDevices(){
        $user = auth()->user();
        if($user->type=='staff'){
            $user=$user->staff;
        }
        $devices = $user->devices()->get();
        return response()->json(['status'=>'success', 'data'=>$devices]);
    }

    public function changeDeviceStatus(Request $request){

        $validator = Validator::make($request->all(), [
            'device_id' => 'required',
            'status' => 'required|in:active,inactive'
        ]);

        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->messages()], 400);
        }

        Device::where('device_unique_id',$request->device_id)->update(['status'=>$request->status]);

        return response()->json(['status'=>'success', 'message'=>'Device status changed successfully']);
    }

    public function inbound(Request $request){
        $validator = Validator::make($request->all(), [
            'from' => 'required',
            'to' => 'required',
            'message' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->messages()], 400);
        }

        $device=Device::where('device_unique_id',$request->to)->first();
        if(!$device){
            return response()->json(['message' => "Device not found"], 400);
        }

     //   $request['from']=getPhoneNumberWithDialCode($request->from);

        $message=new Message();
        $message->customer_id=$device->customer_id;
        $message->body=$request->message;
        $message->numbers=json_encode(['from'=>$request->from,'to'=>[$request->to]]);
        $message->type='inbox';
        $message->read='yes';
        $message->schedule_completed='yes';
        $message->save();

        $messageLog=new MessageLog();
        $messageLog->customer_id=$message->customer_id;
        $messageLog->message_id=$message->id;
        $messageLog->body=$message->body;
        $messageLog->from=$request->from;
        $messageLog->device_unique_id=$device->device_unique_id;
        $messageLog->to=$device->id;
        $messageLog->type='inbox';
        $messageLog->status='succeed';
        $messageLog->save();

        return response()->json(['message' => "Inbound received successfully"]);

    }
}
